%{
This m-file belongs to Chapter 4 of the dissertation 'Electrochemical CO2
conversion with a flow-through copper hollow fibre'
Anne Sustronk (1,2)
(1) Films in Fluids, Department of Science and Technology, University of 
Twente, Enschede, The Netherlands
(2) Photocatalytic Synthesis Group, Department of Science and Technology, 
University of Twente, Enschede, The Netherlands

The file is used to prepare Figure 1in the chapter.

This file is called by Fig1.m and requires CalcErr.m for execution.

Note that the file locations in lines 41, 70, 75 and 80 should be added 
before execution of Fig2And3.m.

Brief overview of experimental conditions: Cu hollow fiber working 
electrode, 0.5M KHCO3 or KOH  in MilliQ water, CO purged through fiber,
Ar purged through head space, chronoamperometry

Note that the flow rate to calculate the partial current densities is set
to 30 ml min-1. This is most likely an underestimation of the actual flow
rate at more negative potentials

Version date m-file: March 22, 2023
%}

function [Data] = ConstantPotentialCO2(Data)
warning('off','MATLAB:table:ModifiedAndSavedVarnames')

% Rename input
Folder1 = Data.Folder1;
Folder2 = Data.Folder2;
FileNameEC = Data.FileNameEC;
Area = Data.Area;
AreaErr = Data.AreaErr;
ESet = Data.ESet;
VarName = Data.VarName;

% Main file folder
Location = {'...\Data Chapter 4 Ethylene'};

% Some constants
Vm = 22.4;              % Molar volume (ml/mmol)
FlowRate = 30/60;       % Flow rate of gas to GC (ml/s)
F = 96485;              % Faraday constant (C/mol)
MwFA = 47;              % Formic acid molecular weight (g/mol)
MwProp = 60.10;         % Molecular weight 1-propanol (g/mol)
Vcat = 50/1000;         % Catholyte/anolyte volume (L)

% Intercept and slope of calibration curve for H2 (Nov22)
b0H2 = -2025.09592;
b1H2 = 294.54225;

% Intercept and slope of calibration curve for CO (Nov22)
b0CO = -4588.95301;
b1CO = 22.26015;

% Intercept and slope of calibration cure for FA (Aug22)
b0FA = 125.67316;
b1FA = 464.91233;

% Intercept and slope of calibration curve for 1-propanol (Nov22)
b0Prop = 421.06;
b1Prop = 814.03;

%% Calibration input

% H2 calibration Nov22
CalH2Data = importdata('...\Data Chapter 4 Ethylene\Matlab\CalH2DataNov22.txt');
XCalH2 = [1000*ones(12,1); 1500*ones(12,1); 2500*ones(12,1); 5000*ones(11,1); 8000*ones(12,1); 10000*ones(12,1); 15000*ones(12,1); 20000*ones(12,1)]';
YCalH2 = [CalH2Data.data(:,1); CalH2Data.data(:,2); CalH2Data.data(:,3); CalH2Data.data(:,4); CalH2Data.data(:,5); CalH2Data.data(:,6); CalH2Data.data(:,7);CalH2Data.data(:,8)]';

% CO
CalCOData = importdata('...\Data Chapter 4 Ethylene\Matlab\CalCODataNov22.txt');
XCalCO = [2500*ones(12,1); 5000*ones(12,1); 8000*ones(12,1); 10000*ones(12,1)]';
YCalCO = [CalCOData.data(:,1); CalCOData.data(:,2); CalCOData.data(:,3); CalCOData.data(:,4)]';

% FA calibration Aug22
CalFAAugData = importdata('...\Data Chapter 4 Ethylene\Matlab\CalFADataAug22.txt');
XCalFAAug22 = [1*ones(2,1);10*ones(2,1); 50*ones(2,1); 100*ones(2,1); 150*ones(2,1)]';
YCalFAAug22 = [CalFAAugData.data(:,1); CalFAAugData.data(:,2); CalFAAugData.data(:,3); CalFAAugData.data(:,4); CalFAAugData.data(:,5)]';

% 1-Propanol
XCalPropNov22 = [10 10 25 25 50 50 75 75 100 100];
YCalPropNov22 = [8609.7 7744.3 20914.3 20714.2 40365.6 40061.3 65905.9 63187.3 80206.1 79795.1];


%% TCD file paths
FileNameTCD = {'\Anne20220208-TCD.txt'};
FilePathTCD = append(Location, Folder1, Folder2, FileNameTCD);
FilePathTCD = cell2mat(FilePathTCD);  % Convert cell array with file paths to matrix

%% FID data file path
FileNameFID = {'\Anne20220208-FID.txt'};
FilePathFID = append(Location, Folder1, Folder2, FileNameFID);
FilePathFID = cell2mat(FilePathFID);  % Convert cell array with file paths to matrix

%% FA data file path
FilePathFA = append(Location, Folder1, '\FA areas.txt');
FilePathFA = cell2mat(FilePathFA);  % Convert cell array with file paths to matrix
LoadDataFA = importdata(FilePathFA);

%% 1-Propanol data file path
FilePathProp = append(Location, Folder1, '\propanol areas.txt');
FilePathProp = cell2mat(FilePathProp);  % Convert cell array with file paths to matrix
LoadDataProp = importdata(FilePathProp);

%% File path resistance values
FilePathR =  append(Location, Folder1,'\resistances.txt');
FilePathR = cell2mat(FilePathR);  % Convert cell array with file paths to matrix
LoadDataResistance = importdata(FilePathR);

%% Calculations

for j = 1:length(ESet)

    FilePathEC = append(Location, Folder1, Folder2(j,:), FileNameEC(j,:)); %Retrieve file path potentiostat data
    FilePathEC = cell2mat(FilePathEC);  % Convert cell array with file paths to matrix

    %% Load data from potentiostat
    LoadDataEC = importdata(FilePathEC);

    % LoadDataEC is a struct containing fields for every investigated potential
    % The columns refer to: time/s, <I>/mA, Ewe/V, <Ece>/V, dQ/C, Rcmp/Ohm,
    % P/W, Ewe-Ece/V
    Data.(VarName(j,:)).EC = LoadDataEC.data;

    % Recalculate time (time does not start at t=0)
    tElapsed = Data.(VarName(j,:)).EC(:,1);
    tStart = Data.(VarName(j,:)).EC(1,1);
    Data.(VarName(j,:)).t = tElapsed - tStart;

    % Total current
    Data.(VarName(j,:)).I = Data.(VarName(j,:)).EC(:,2);

    % Calculate current density
    Data.(VarName(j,:)).CD = Data.(VarName(j,:)).EC(:,2)/Area(j);

    % Calculate average current (density) over the last 30 minutes of the
    % experiment
    findTime = find(Data.(VarName(j,:)).t >= 30*60);

    q1 = size(findTime);

    % Average current and estimation of standard deviation
    Data.Iavg30mins(j,1) = mean(Data.(VarName(j,:)).I(findTime(:,1),1));
    Data.Iavg30mins(j,2) = sqrt(sum((Data.(VarName(j,:)).I(findTime(:,1),1)-Data.Iavg30mins(j,1)).^2)/(q1(1,1)-1));

    % Average current density and estimation of standard deviation
    Data.CDavg30mins(j,1) = mean(Data.(VarName(j,:)).CD(findTime(:,1),1));
    Data.CDavg30mins(j,2) = sqrt((Data.Iavg30mins(j,2)/Data.Iavg30mins(j,1))^2 + ((AreaErr(j)/Area(j))^2))*Data.CDavg30mins(j,1);

    %% Obtain H2 current and H2 partial current density
    LoadDataTCD = readtable(FilePathTCD(j,:));
    LoadDataTCD = LoadDataTCD(:,7:end);

    Data.(VarName(j,:)).TCD = LoadDataTCD;

    TCDArea = table2array(Data.(VarName(j,:)).TCD);

    % Calculate average area of injections 5, 6 and 7
    Data.TCD_avg(j,:) = mean(TCDArea(5:end,:));

    % Obtain H2 concentration and error in H2 concentration
    Width = size(Data.(VarName(j,:)).TCD(:,1));
    q = Width(1,1)-4; % Determine amount of values used to calculate the average
    Data.ConcAndErr(j,:) = CalcErr(XCalH2, YCalH2, q,b0H2,b1H2,Data.TCD_avg(j,1));

    % Calculate corresponding current
    Data.IH2(j,:) = ((((Data.ConcAndErr(j,:))/10000/100)*(FlowRate/Vm))*-2*F);

    % Calculate corresponding current density
    Data.CDH2(j,1) = Data.IH2(j,1)/Area(j);

    % Calculate error in current density
    Data.CDH2(j,2) = sqrt(((Data.IH2(j,2))/Data.IH2(j,1))^2+(AreaErr(j)/Area(j))^2)*(Data.CDH2(j,1));

    %% Obtain CO current and CO partial current density

    % Obtain CO concentration and error in CO concentration
    Width = size(Data.(VarName(j,:)).TCD(:,4));
    qCO = Width(1,1)-4; % Determine amount of values used to calculate the average

    Data.ConcAndErrCO(j,:) = CalcErr(XCalCO, YCalCO, qCO,b0CO,b1CO,Data.TCD_avg(j,4));

    % Calculate corresponding current
    Data.ICO(j,:) = ((((Data.ConcAndErrCO(j,:))/10000/100)*(FlowRate/Vm))*-2*F);

    % Calculate corresponding current density
    Data.CDCO(j,1) = Data.ICO(j,1)/Area(j);

    % Calculate error in current density
    Data.CDCO(j,2) = sqrt(((Data.ICO(j,2))/Data.ICO(j,1))^2+(AreaErr(j)/Area(j))^2)*(Data.ICO(j,1));


    %% Obtain hydrocarbons current and hydrocarbons partial current density

    LoadDataFID = readtable(FilePathFID(j,:));
    LoadDataFID = LoadDataFID(:,7:end);
    Data.(VarName(j,:)).FID = LoadDataFID;

    XCal = [933 931 957 958 967];       % ppm methane, ethylene, ethane, propylene, propane
    YCal = [20915; 39461; 41438; 60924; 61100]; % Peak area calibration
    StDevCal = [676.6660416	1266.426098	1330.122377	2000.205671	1940.911142]'; % Standard deviation of peak area
    n = [8 12 14 18 20];      % Amount of electrons involved for converting CO in methane, ethene, ethane, etc.

    FIDArea = table2array(Data.(VarName(j,:)).FID);

    % Calculate average area of injections 5, 6 and 7
    Data.FID_avg(j,:) = mean(FIDArea(5:end,1:5));

    % Determine amount of values used to calculate the average of
    % injections
    Width = size(Data.(VarName(j,:)).FID(:,1));
    q = Width(1,1)-4; % Determine amount of values used to calculate the average

    % Estimate standard deviation of injections 5, 6, 7
    Data.FID_avgstdev(j,:) = sqrt(sum((FIDArea(5:end,1:5)-Data.FID_avg(j,:)).^2)/(q-1));

    % Calculate current (mA) and current density (mA cm^-2)
    Data.FIDppm(j,:) = Data.FID_avg(j,:).*XCal./YCal';
    Data.FID_I(j,:) = ((((Data.FIDppm(j,:))/10000/100)*(FlowRate/Vm)).*-n*F);
    Data.FID_CD(j,:) = Data.FID_I(j,:)/Area(j);

    % Estimate error in current and current density
    for i = 1:5
        Data.FIDppmErr(j,i) = Data.FIDppm(j,i)*sqrt((Data.FID_avgstdev(j,i)/Data.FID_avg(j,i)).^2+(StDevCal(i,:)/YCal(i,:)).^2);
        Data.FID_I_Err(j,i)=((((Data.FIDppmErr(j,i))/10000/100)*(FlowRate/Vm)).*-n(i)*F);

        Data.FID_CD_Err(j,i) = sqrt(((Data.FID_I_Err(j,i))/Data.FID_I(j,i))^2+(AreaErr(j)/Area(j))^2)*(Data.FID_CD(j,i));
    end

    %% Obtain formate current and formate partial current density
    Data.FA(j,:) = LoadDataFA.data(j,:);

    Data.ConcAndErrFACat(j,:) = CalcErr(XCalFAAug22,YCalFAAug22,1,b0FA,b1FA,Data.FA(j,1));   % FA concentration in catholyte + error
    Data.ConcAndErrFAAn(j,:) = CalcErr(XCalFAAug22,YCalFAAug22,1,b0FA,b1FA,Data.FA(j,2));    % FA concentration in anolyte + error

    if Data.ConcAndErrFACat(j,1) > 0  & Data.ConcAndErrFAAn(j,1) > 0
        Data.ConcAndErrFA(j,1) = Data.ConcAndErrFACat(j,1) + Data.ConcAndErrFAAn(j,1);
        Data.ConcAndErrFA(j,2) =sqrt((Data.ConcAndErrFACat(j,2))^2 + (Data.ConcAndErrFAAn(j,2))^2);
    elseif Data.ConcAndErrFACat(j,1) > 0
        Data.ConcAndErrFA(j,1) = Data.ConcAndErrFACat(j,1);
        Data.ConcAndErrFA(j,2) = Data.ConcAndErrFACat(j,2);

    else    % This means only Data.ConcAndErrFAAn(j,:) > 0
        Data.ConcAndErrFA(j,1) = Data.ConcAndErrFAAn(j,1);
        Data.ConcAndErrFA(j,2) = Data.ConcAndErrFAAn(j,2);

    end

    % Calculate corresponding current
    Data.IFA(j,:) = (((Data.ConcAndErrFA(j,:))/1000/MwFA)*Vcat*-2*F)/3600*1000;

    % Calculate corresponding current density
    Data.CDFA(j,1) = Data.IFA(j,1)/Area(j);

    % Calculate error in current density
    Data.CDFA(j,2) = sqrt(((Data.IFA(j,2))/Data.IFA(j,1))^2+(AreaErr(j)/Area(j))^2)*(Data.CDFA(j,1));

    %% Obtain 1-propanol partial current density

    Data.Prop(j,:) = LoadDataProp.data(j,:);

    Data.ConcAndErrPropCat(j,:) = CalcErr(XCalPropNov22,YCalPropNov22,1,b0Prop,b1Prop,Data.Prop(j,1));   % Prop concentration in catholyte + error

    Data.ConcAndErrProp(j,1) = Data.ConcAndErrPropCat(j,1);
    Data.ConcAndErrProp(j,2) = Data.ConcAndErrPropCat(j,2);


    % Calculate corresponding current
    Data.IProp(j,:) = (((Data.ConcAndErrProp(j,:))/1000/MwProp)*Vcat*-18*F)/3600*1000;

    % Calculate corresponding current density
    Data.CDProp(j,1) = Data.IProp(j,1)/Area(j);

    % Calculate error in current density
    Data.CDProp(j,2) = sqrt(((Data.IProp(j,2))/Data.IProp(j,1))^2+(AreaErr(j)/Area(j))^2)*(Data.CDProp(j,1));

    %% Calculate actual potential at working electrode and error in the potential

    Data.R(j,:) = LoadDataResistance(j,:);

    RAvg = mean(Data.R(j,:));

    % Actual potentail
    Data.Ereal(j,1) = ESet(j)- mean((Data.(VarName(j,:)).EC(findTime(:,1),2))/1000)*RAvg;

    % Error calculations
    RErr = Data.R(j,1) - RAvg;

    %Error in potential
    Data.Ereal(j,2) =  mean((Data.(VarName(j,:)).EC(findTime(:,1),2))/1000)*RErr;


    %% Calculate current that is not assigned

    % Convert all NaN in 0
    IndexFA = isnan(Data.CDFA);
    Data.CDFA(IndexFA) = 0;

    IndexProp = isnan(Data.CDProp);
    Data.CDProp(IndexProp) = 0;

    b = size(Data.FID_CD);

    for i = 1:b(1,2)
        IndexFID = isnan(Data.FID_CD(:,i));
        Data.FID_CD(IndexFID,i) = 0;
    end

    Data.Unaccounted(j,1) = Data.CDavg30mins(j,1) - Data.CDH2(j,1) - Data.CDCO(j,1) - Data.CDFA(j,1) - Data.CDProp(j,1) - Data.FID_CD(j,2) - Data.FID_CD(j,3) - Data.FID_CD(j,4);
end
end